// JavaScript pour le dashboard

document.addEventListener("DOMContentLoaded", () => {
  initDashboard()
  initSearch()
  initNavigation()
  initGenreFilters()
})

function initDashboard() {
  console.log("Dashboard initialisé")

  // Charger les données utilisateur
  loadUserData()

  // Initialiser les interactions
  initCardInteractions()
  initFavoriteButtons()
}

function initSearch() {
  const searchInput = document.getElementById("searchInput")
  if (searchInput) {
    let searchTimeout

    searchInput.addEventListener("input", function () {
      clearTimeout(searchTimeout)
      searchTimeout = setTimeout(() => {
        performSearch(this.value)
      }, 300)
    })
  }
}

function performSearch(query) {
  if (query.length < 2) {
    resetSearch()
    return
  }

  // Appel AJAX pour la recherche
  fetch(`api/search.php?q=${encodeURIComponent(query)}`)
    .then((response) => response.json())
    .then((data) => {
      displaySearchResults(data)
    })
    .catch((error) => {
      console.error("Erreur de recherche:", error)
    })
}

function displaySearchResults(results) {
  const cardsContainer = document.getElementById("popularCards")
  if (!cardsContainer) return

  cardsContainer.innerHTML = ""

  results.forEach((performance) => {
    const cardHTML = createPerformanceCard(performance)
    cardsContainer.appendChild(cardHTML)
  })
}

function createPerformanceCard(performance) {
  const cardItem = document.createElement("div")
  cardItem.className = "card-item"
  cardItem.onclick = () => viewPerformance(performance.id)

  cardItem.innerHTML = `
        <div class="card" style="background-image: url('${performance.poster_url || "assets/images/placeholder.jpg"}')"></div>
        <div class="card-info">
            <div class="title">${performance.title}</div>
            <div class="rating">
                ★ ${Number.parseFloat(performance.avg_rating || 0).toFixed(1)} 
                (${performance.review_count || 0})
            </div>
        </div>
    `

  return cardItem
}

function resetSearch() {
  // Recharger les spectacles populaires
  location.reload()
}

function initNavigation() {
  const navItems = document.querySelectorAll(".nav-item")

  navItems.forEach((item) => {
    item.addEventListener("click", function () {
      const page = this.dataset.page
      navigateToPage(page)
    })
  })
}

function navigateToPage(page) {
  // Marquer l'élément actif
  document.querySelectorAll(".nav-item").forEach((item) => {
    item.classList.remove("active")
  })

  document.querySelector(`[data-page="${page}"]`).classList.add("active")

  // Navigation
  switch (page) {
    case "dashboard":
      window.location.href = "dashboard.php"
      break
    case "favorites":
      window.location.href = "favorites.php"
      break
    case "upcoming":
      window.location.href = "upcoming.php"
      break
    case "categories":
      window.location.href = "categories.php"
      break
    case "friends":
      window.location.href = "friends.php"
      break
    case "groups":
      window.location.href = "groups.php"
      break
  }
}

function initGenreFilters() {
  const genres = document.querySelectorAll(".genre")

  genres.forEach((genre) => {
    genre.addEventListener("click", function () {
      const category = this.dataset.category
      filterByCategory(category)
    })
  })
}

function filterByCategory(category) {
  // Marquer le genre actif
  document.querySelectorAll(".genre").forEach((g) => g.classList.remove("active"))
  document.querySelector(`[data-category="${category}"]`).classList.add("active")

  // Filtrer les spectacles
  fetch(`api/performances.php?category=${category}`)
    .then((response) => response.json())
    .then((data) => {
      displaySearchResults(data)
    })
    .catch((error) => {
      console.error("Erreur de filtrage:", error)
    })
}

function initCardInteractions() {
  const cards = document.querySelectorAll(".card-item")

  cards.forEach((card) => {
    card.addEventListener("mouseenter", function () {
      this.style.transform = "translateY(-5px)"
    })

    card.addEventListener("mouseleave", function () {
      this.style.transform = "translateY(0)"
    })
  })
}

function initFavoriteButtons() {
  const favoriteButtons = document.querySelectorAll(".btn-fav")

  favoriteButtons.forEach((button) => {
    button.addEventListener("click", function (e) {
      e.stopPropagation()
      const performanceId = this.dataset.performanceId
      toggleFavorite(performanceId)
    })
  })
}

function viewPerformance(id) {
  window.location.href = `performance.php?id=${id}`
}

function toggleFavorite(performanceId) {
  fetch("api/favorites.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      performance_id: performanceId,
      action: "toggle",
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        updateFavoriteButton(performanceId, data.is_favorite)
        showNotification(data.is_favorite ? "Ajouté aux favoris" : "Retiré des favoris", "success")
      } else {
        showNotification("Erreur lors de la mise à jour", "error")
      }
    })
    .catch((error) => {
      console.error("Erreur:", error)
      showNotification("Erreur de connexion", "error")
    })
}

function updateFavoriteButton(performanceId, isFavorite) {
  const button = document.querySelector(`[data-performance-id="${performanceId}"]`)
  if (button) {
    button.classList.toggle("active", isFavorite)
  }
}

function showTrailer(performanceId) {
  // Ouvrir la bande-annonce dans une modal
  const modal = document.createElement("div")
  modal.className = "trailer-modal"
  modal.innerHTML = `
        <div class="modal-content">
            <span class="close-modal">&times;</span>
            <iframe src="api/trailer.php?id=${performanceId}" frameborder="0" allowfullscreen></iframe>
        </div>
    `

  document.body.appendChild(modal)

  modal.querySelector(".close-modal").addEventListener("click", () => {
    modal.remove()
  })

  modal.addEventListener("click", (e) => {
    if (e.target === modal) {
      modal.remove()
    }
  })
}

function logout() {
  if (confirm("Êtes-vous sûr de vouloir vous déconnecter ?")) {
    window.location.href = "api/logout.php"
  }
}

function loadUserData() {
  // Charger les données utilisateur depuis l'API
  fetch("api/user.php")
    .then((response) => response.json())
    .then((data) => {
      updateUserProfile(data)
    })
    .catch((error) => {
      console.error("Erreur de chargement des données utilisateur:", error)
    })
}

function updateUserProfile(userData) {
  const profileName = document.querySelector(".profile .info div:first-child")
  const profileHandle = document.querySelector(".profile .info div:last-child")
  const profileAvatar = document.querySelector(".profile img")

  if (profileName) profileName.textContent = userData.display_name
  if (profileHandle) profileHandle.textContent = `@${userData.handle}`
  if (profileAvatar && userData.avatar_url) profileAvatar.src = userData.avatar_url
}

function showNotification(message, type) {
  const notification = document.createElement("div")
  notification.className = `notification ${type}`
  notification.textContent = message
  document.body.appendChild(notification)

  setTimeout(() => {
    notification.remove()
  }, 3000)
}
